﻿using System;
using IIscAppLoaderInterfaces;
using log4net;
using SampleApp.Serial;

namespace SampleApp
{
  public class IscApp : IIscApp
  {
    private static readonly ILog Logger = LogManager.GetLogger(typeof (IscApp));

    private IIscAppConnector _appHost;

    private SerialPortUsageExample _serialPortUsageExample;

    void IIscApp.Init(IIscAppConnector iscAppHost)
    {
      // Put code to initialize stuff managed by this app here.
      // No KNX communication is allowed!
      // Return true if the app is initialized and ready to run.
      // Return false if not ready to run.
      Logger.Info("Initializing SampleApp.");
      
      _appHost = iscAppHost;

      Logger.Info("Initializing done.");
    }

    void IIscApp.Run()
    {
      // Put code to get app running here.
      // If values on the KNX bus should be initialized, do this right at the beginning.
      // Return true if the app is running, false if not.

      // Example to write to the KNX bus using differenet data point types
      WriteToKnxBus();

      // Example how to read parameter values
      ReadParametersValue();

      // Example how to create a file on the user partition
      UserPartitionUsageExample.CreateFileOnUserPartition(_appHost);

      // Example indicate SD card and getting it's path if it is present
      SdCardUsageExample.UseSdCard(_appHost);

      // Check Serial port and write it's name if present
      CheckSerialPort();

      // Initialize serial port usage example
      _serialPortUsageExample = new SerialPortUsageExample(_appHost);

      Logger.Info("Config root path: " + _appHost.ConfigRootPath);
      for (uint i = 1; i <= 10; i++)
      {
        Logger.Info("Text Parameter " + i + ": " + _appHost.GetTextParameter(i));
        Console.WriteLine("Text Parameter " + i + ": " + _appHost.GetTextParameter(i));
      }
      for (uint i = 1; i <= 10; i++)
      {
        Logger.Info("IP Parameter " + i + ": " + _appHost.GetIpParameter(i));
        Console.WriteLine("IP Parameter " + i + ": " + _appHost.GetIpParameter(i));
      }
      for (uint i = 1; i <= 10; i++)
      {
        Logger.Info("Number Parameter " + i + ": " + _appHost.GetNumberParameter(i));
        Console.WriteLine("Number Parameter " + i + ": " + _appHost.GetNumberParameter(i));
      }

      Logger.Info("SampleApp running.");
    }

    void IIscApp.Exit()
    {
      // Put code here that needs to be done to shut-down this app.
      Logger.Info("SampleApp exiting.");
    }

    object IIscApp.GetValue(uint coId)
    {
      //This method will only be called in a future version of Programmable, but is already part of the interface.
      //You should currently implement it by throwing NotImplementedException:
      throw new NotImplementedException();
    }

    /// <summary>
    ///   Called by the host when an interesting value changed on KNX.
    /// </summary>
    /// <param name="coId">The app-specific type (from 1 to 64) of the event</param>
    /// <param name="value">The current value from the KNX event. Its type depends on the KNX type of the value.</param>
    void IIscApp.OnValueReceived(uint coId, object value)
    {
      Logger.Info(string.Format("Received new value for CO {0}: {1}", coId, value));
      // Put code here to react to a new value of a CO.
      switch (coId)
      {
        // Writing on the communication object number 46 will connect the serial port and writes the value on the bus          
        case 54:
          // Example how to use the serial port. Precondition of this example is a loop back serial connection          
          _serialPortUsageExample.CommunicateWithSerialPort(string.Format("{0}", value));
          return;
      }

      // currently, just write back
      _appHost.WriteValue(coId + 30u, value);
    }

    private void CheckSerialPort()
    {
      // Check if the serial port is present
      var isSerialPortPresent = _appHost.IsSerialPortPresent;
      if (isSerialPortPresent == false)
      {
        _appHost.WriteValue(46, "No serial port");
        return;
      }

      // Get the name of the serial port
      var serialPortName = _appHost.SerialPortName;
      _appHost.WriteValue(46, serialPortName);
    }

    private void ReadParametersValue()
    {
      // Get the value of the IP parameter with the parameter ID 1
      var ipParameter = _appHost.GetIpParameter(1);

      _appHost.WriteValue(46, ipParameter.ToString());

      // Get the value of the number parameter with the parameter ID 1
      var numberParameter = _appHost.GetNumberParameter(1);
      _appHost.WriteValue(46, numberParameter.ToString());

      // Get the value of the text parameter with the parameter ID 1
      var textParameter = _appHost.GetTextParameter(1);
      _appHost.WriteValue(46, textParameter);
    }

    private void WriteToKnxBus()
    {
      var dataPointsUsageExampleClass = new DataPointsUsageExample(_appHost);
      dataPointsUsageExampleClass.WriteOneBit();
      dataPointsUsageExampleClass.WriteTwoBits();
      dataPointsUsageExampleClass.WriteFourBits();
      dataPointsUsageExampleClass.WriteOneByte();
      dataPointsUsageExampleClass.WriteTwoBytes();
      dataPointsUsageExampleClass.WriteThreeBytes();
      dataPointsUsageExampleClass.WriteFourBytes();
      dataPointsUsageExampleClass.WriteFourteenBytes();
      dataPointsUsageExampleClass.WriteExceptions();
    }
  }
}