#!/bin/sh
source "/opt/gira/bin/environment"

TAG="ExtendedLoggingSwitcher"

if [ "$1" == "1" ]; then
  LOGMODE="Extended"
elif [ "$1" == "0" ]; then
  LOGMODE="Normal"
else
  logger -t "${TAG}" "Invalid log mode requested: '$1'"
  exit 1
fi

logit()
{
  logger -t "${TAG}" "$1"
}

copy()
{
  # sanity check
  if [ "$#" != "2" ]; then
    logit "Invalid usage of backup()."
    return 1
  fi
  
  cp "$1" "$2"
  if [ "$?" != "0" ]; then
    logit "Failed to backup '$1' to '$2'"
    return 1
  fi
  
  return 0
}


setlog4netconfig()
{
  # sanity check
  if [ "$#" != "1" ]; then
    logit "Invalid usage of setlog4netconfig()."
    return 1
  fi
  
  if [ $1 == "Extended" ]; then
    ln -sf /opt/gira/etc/iscapphost/log4net_debug.config.linux /opt/userdata/etc/iscapphost/log4net.config
    if [ "$?" != "0" ]; then
      logit "Failed to modify 'log4net.config'!"
      return 1
    fi
  elif [ $1 == "Normal" ]; then
    ln -sf /opt/gira/etc/iscapphost/log4net_default.config.linux /opt/userdata/etc/iscapphost/log4net.config
    if [ "$?" != "0" ]; then
      logit "Failed to modify 'log4net.config'!"
      return 1
    fi
  else
    logit "Invalid Log Mode provided to setlog4netconfig(): $1."
    return 1
  fi
  
  return 0
}

sedlog4cplusconfig()
{
  # sanity check
  if [ "$#" != "3" ]; then
    logit "Invalid usage of sedlog4cplusconfig()."
    return 1
  fi
  
  if [ $3 == "Extended" ]; then
    sed -e "/log4cplus.rootLogger/s/INFO/DEBUG/;/LFP.Threshold/s/ERROR/DEBUG/" "$1" > "$2"
    if [ "$?" != "0" ]; then
      logit "Failed to modify '$1'!"
      return 1
    fi
  elif [ $3 == "Normal" ]; then
    sed -e "/log4cplus.rootLogger/s/DEBUG/INFO/;/LFP.Threshold/s/DEBUG/ERROR/" "$1" > "$2"
    if [ "$?" != "0" ]; then
      logit "Failed to modify '$1'!"
      return 1
    fi
  else
    logit "Invalid Log Mode provided to sedlog4cplusconfig(): $3."
    return 1
  fi
  
  return 0
}

seddevicestackconfig()
{
  # sanity check
  if [ "$#" != "3" ]; then
    logit "Invalid usage of seddevicestackconfig()."
    return 1
  fi
  
  if [ $3 == "Extended" ]; then
    sed -e "/ApplicationLogging/s/Level=\"3\"/Level=\"6\"/" "$1" > "$2"
    if [ "$?" != "0" ]; then
      logit "Failed to modify '$1'!"
      return 1
    fi
  elif [ $3 == "Normal" ]; then
    sed -e "/ApplicationLogging/s/Level=\"6\"/Level=\"3\"/" "$1" > "$2"
    if [ "$?" != "0" ]; then
      logit "Failed to modify '$1'!"
      return 1
    fi
  else
    logit "Invalid Log Mode provided to sedlog4cplusconfig(): $3."
    return 1
  fi
  
  return 0
}

logit "${LOGMODE} log mode requested."

# delete old BAKs and TMPs
rm -f /tmp/iscapphostconfig.*
rm -f /tmp/knxstackconfig.*
rm -f /tmp/devicestackconfig.*

TMPISCAPPHOSTCONFIG="$(mktemp -p /tmp iscapphostconfig.XXXXXXXX)"
BAKISCAPPHOSTCONFIG="$(mktemp -p /tmp iscapphostconfig.XXXXXXXX)"
TMPKNXSTACKCONFIG="$(mktemp -p /tmp knxstackconfig.XXXXXXXX)"
BAKKNXSTACKCONFIG="$(mktemp -p /tmp knxstackconfig.XXXXXXXX)"
TMPDEVICESTACKCONFIG="$(mktemp -p /tmp devicestackconfig.XXXXXXXX)"
BAKDEVICESTACKCONFIG="$(mktemp -p /tmp devicestackconfig.XXXXXXXX)"

# back-up configuration file
copy "${ISCAPPHOSTCONFIG}" "${BAKISCAPPHOSTCONFIG}" || exit 1
copy "${KNXSTACKCONFIG}" "${BAKKNXSTACKCONFIG}" || exit 1
copy "${DEVICESTACKCONFIG}" "${BAKDEVICESTACKCONFIG}" || exit 1
  
# stop devicestack, knxstack and iscapphost to allow clean-up of log files
logit "Stopping application..."
killall -9 iscapphost knxstack devicestack
  
if [ "${LOGMODE}" == "Extended" ]; then
  logit "Enabling extended logging mode..."
else
  logit "Enabling extended logging mode..."
fi
  
setlog4netconfig "${LOGMODE}" || exit 1
sedlog4cplusconfig "${BAKISCAPPHOSTCONFIG}" "${TMPISCAPPHOSTCONFIG}" "${LOGMODE}" || exit 1
sedlog4cplusconfig "${BAKKNXSTACKCONFIG}" "${TMPKNXSTACKCONFIG}" "${LOGMODE}" || exit 1
seddevicestackconfig "${BAKDEVICESTACKCONFIG}" "${TMPDEVICESTACKCONFIG}" "${LOGMODE}" || exit 1

if [ "${LOGMODE}" == "Extended" ]; then
  # update time stamp to allow disabling debug after a specified time
  touch "${DEBUGACTIVE}"
  
  # remove old logs
  logit "Cleaning-up logs..."
  rm -f "${DEVICESTACKLOG}" \
        "${KNXSTACKLOG}" \
        "${KNXSTACKLOGPERSISTENT}" \
        "${ISCAPPHOSTLOG}" \
        "${ISCAPPHOSTLOGPERSISTENT}"
else
  # remove debug mode indicator
  rm -f "${DEBUGACTIVE}"
fi

# remove old configs
rm "${ISCAPPHOSTCONFIG}"
rm "${KNXSTACKCONFIG}"
rm "${DEVICESTACKCONFIG}"

# replace config files
RESTOREOLD="false"
EMERGENCY="false"
FATAL="false"

copy "${TMPISCAPPHOSTCONFIG}" "${ISCAPPHOSTCONFIG}" || RESTOREOLD="true"

if [ "${RESTOREOLD}" != "true" ]; then
  copy "${TMPKNXSTACKCONFIG}" "${KNXSTACKCONFIG}" || RESTOREOLD="true"
fi

if [ "${RESTOREOLD}" != "true" ]; then
  copy "${TMPDEVICESTACKCONFIG}" "${DEVICESTACKCONFIG}" || RESTOREOLD="true"
fi

if [ "${RESTOREOLD}" == "true" ]; then
  rm "${ISCAPPHOSTCONFIG}"
  rm "${KNXSTACKCONFIG}"
  rm "${DEVICESTACKCONFIG}"
  
  copy "${BAKISCAPPHOSTCONFIG}" "${ISCAPPHOSTCONFIG}" || EMERGENCY="true"
  copy "${BAKKNXSTACKCONFIG}" "${KNXSTACKCONFIG}" || EMERGENCY="true"
  copy "${BAKDEVICESTACKCONFIG}" "${DEVICESTACKCONFIG}" || EMERGENCY="true"
fi

if [ "${EMERGENCY}" == "true" ]; then
  logit "!!! While switching to ${LOGMODE} Logging, a severe failure has occurred and the system is damaged. !!!"
fi

# restart system to allow changes to take effect
# A devicestack restart is possible but not recommended by the GDS team.
logit "Restarting sytem..."
reboot
