/******************************************************************************
*
* File: BIM_M13x.h
*
* Description: Include file for BIM M 13x
*
* Date: 01/31/2007
*
* Revision: 1
*
* (C) 2007 Copyright Siemens AG
*
* Author: Schiekofer
*
* Modified:
*
* Changes:
*
******************************************************************************/

#ifndef __BIM_M13X_H
#define __BIM_M13X_H

#include "user.h"		

#include <intrinsics.h>         
#ifdef BIM_M130
  #include <io78f0534_64.h>   
#endif
#ifdef BIM_M131
  #include <io78f0535_64.h>   
#endif
#ifdef BIM_M132
  #include <io78f0537_64.h>   
#endif
#define SECTOR_SIZE                 1024

//-----------------------------------------------------------------------------
// generic defines
//-----------------------------------------------------------------------------
#define BIT0                (0x01)
#define BIT1                (0x02)
#define BIT2                (0x04)
#define BIT3                (0x08)
#define BIT4                (0x10)
#define BIT5                (0x20)
#define BIT6                (0x40)
#define BIT7                (0x80)

#define TRUE            1
#define FALSE           0
#define ON              TRUE
#define OFF             FALSE
#define NULL            0

#define set_bit         |=                                  //!< macro for 'bitwise-OR' to set a bit
#define reset_bit       &= ~                                //!< macro for 'bitwise-AND-NOT' to reset a bit
#define toggle_bit      ^=                                  //!< macro for 'bitwise-XOR' to toggle a bit

#define Word(X,Y)       (((unsigned short)(X)<<8)|(Y))      //!< generate word from 2 byte
#define Swap(word)      (((word)>>8) + (((BYTE)(word))<<8)) //!< swaps bytes of a word (big-/little-endian)
#define HIBYTE(x)       (BYTE)*((BYTE*)(&(x))+1)            //!< high byte of 16 bit int (intel convention)
#define LOBYTE(x)       (BYTE)*((BYTE*)&(x))                //!< low  byte of 16 bit int (intel convention)
#define TRIGGERWATCHDOG()   WDTE = 0xAC

//-----------------------------------------------------------------------------------------------------------
// timer values
//-----------------------------------------------------------------------------------------------------------
#define HOUR1       (8640000)
#define MIN1        (HOUR1/60)
#define SEC10       (HOUR1/360)
#define SEC6        (HOUR1/600)
#define SEC3        (HOUR1/1200)
#define SEC1        (HOUR1/3600)
#define MSEC500     (HOUR1/7200)
#define MSEC200     (HOUR1/18000)
#define MSEC100     (HOUR1/36000)
#define MSEC30      (HOUR1/120000)
#define MSEC20      (HOUR1/180000)

//-----------------------------------------------------------------------------------------------------------
// properties
//-----------------------------------------------------------------------------------------------------------
#define PDT_CONTROL                             0x00
#define PDT_CHAR                                0x01
#define PDT_UNSIGNED_CHAR                       0x02
#define PDT_INT                                 0x03
#define PDT_UNSIGNED_INT                        0x04
#define PDT_EIB_FLOAT                           0x05
#define PDT_DATE                                0x06
#define PDT_TIME                                0x07
#define PDT_LONG                                0x08
#define PDT_UNSIGNED_LONG                       0x09
#define PDT_FLOAT                               0x0A
#define PDT_DOUBLE                              0x0B
#define PDT_CHAR_BLOCK                          0x0C
#define PDT_POLL_GROUP_SETTINGS                 0x0D
#define PDT_SHORT_CHAR_BLOCK                    0x0E
#define PDT_GENERIC_01                          0x11
#define PDT_GENERIC_02                          0x12
#define PDT_GENERIC_03                          0x13
#define PDT_GENERIC_04                          0x14
#define PDT_GENERIC_05                          0x15
#define PDT_GENERIC_06                          0x16
#define PDT_GENERIC_07                          0x17
#define PDT_GENERIC_08                          0x18
#define PDT_GENERIC_09                          0x19
#define PDT_GENERIC_10                          0x1A
#define PDT_GENERIC_11                          0x1B
#define PDT_GENERIC_12                          0x1C
#define PDT_GENERIC_13                          0x1D
#define PDT_GENERIC_14                          0x1E
#define PDT_GENERIC_15                          0x1F
#define PDT_GENERIC_16                          0x20
#define PDT_GENERIC_17                          0x21
#define PDT_GENERIC_18                          0x22
#define PDT_GENERIC_19                          0x23
#define PDT_GENERIC_20                          0x24
#define PDT_VERSION                             0x30
#define PDT_ALARM_INFO                          0x31
#define PDT_BINARY_INFORMATION                  0x32
#define PDT_BITSET8                             0x33
#define PDT_BITSET16                            0x34
#define PDT_ENUM8                               0x35
#define PDT_SCALING                             0x36
#define PDT_FUNCTION                            0x3E

#define PDT_WR_EN                               0x80
#define FUNCTION_FLAG                           0x40

//-----------------------------------------------------------------------------------------------------------
// Ramflags
//-----------------------------------------------------------------------------------------------------------
#define IDLEERROR                   0x01
#define TRANSMITTING                0x02
#define TRANSMIT_REQUEST            0x03
#define DATA_REQUEST                0x04
#define UPDATE                      0x08
#define SET_FLG                     0x80
#define CLR_FLG                     0x00

//-----------------------------------------------------------------------------------------------------------
// ConfigByte
//-----------------------------------------------------------------------------------------------------------
#define TX_SYSTEM                   0x00
#define TX_URGENT                   0x01
#define TX_NORMAL                   0x02
#define TX_LOW                      0x03
#define COMM_ENABLE                 0x04
#define READ_ENABLE                 0x08
#define WRITE_ENABLE                0x10
#define TRANSMIT_ENABLE             0x40
#define UPDATE_ON_RESPONSE          0x80
#define VALID_BCU2ADR               NULL

//-----------------------------------------------------------------------------------------------------------
// Typebyte
//-----------------------------------------------------------------------------------------------------------
#define UINT1                       0x00                    // 1 bit
#define UINT2                       0x01                    // 2 bit
#define UINT3                       0x02                    // 3 bit
#define UINT4                       0x03                    // 4 bit
#define UINT5                       0x04                    // 5 bit
#define UINT6                       0x05                    // 6 bit
#define UINT7                       0x06                    // 7 bit
#define UINT8                       0x07                    // 8 bit
#define UINT16                      0x08                    // 16 bit
#define TIME_DATE                   0x09                    // 3byte
#define FLOAT                       0x0A                    // 4byte
#define DATA6                       0x0B                    // 6byte
#define DOUBLE                      0x0C                    // 8byte
#define DATA10                      0x0D                    // 10byte
#define MAXDATA                     0x0E                    // 14byte
#define VARDATA                     0x0F                    // 1-14 bytes ( not supported by standard callback)

//-----------------------------------------------------------------------------------------------------------
// services
//-----------------------------------------------------------------------------------------------------------
#define SERVICE_MASK            0x3C
#define LAYER_MASK              0xC0
#define TYPE_MASK               0x03

// message types (2 bit)
#define REQ                     0
#define CON                     1
#define IND                     2

// service types (4 bit)
#define GROUP_DATA	            (0 << 2)
#define DATA_CO			        (1 << 2)
#define DATA_CL                 (2 << 2)
#define BROADCAST		        (3 << 2)
#define CONNECTION_TIMEOUT      (4 << 2)
#define ACKNOWLEDGE_TIMEOUT     (5 << 2)
#define VALUE_WRITE             (6 << 2)
#define VALUE_READ              (7 << 2)
#define VALUE_RESP              (8 << 2)
#define DISCONNECT              (9 << 2)

// layer bits (2 bit)
#define AL                      (0 << 6)
#define TL                      (1 << 6)
#define NL                      (2 << 6)
#define LL                      (3 << 6)

// service definition
// LL
#define L_GROUP_DATA_requ		LL + GROUP_DATA + REQ
#define L_DATA_requ			    LL + DATA_CO    + REQ
#define L_BROADCAST_requ		LL + BROADCAST  + REQ

#define L_GROUP_DATA_ind        LL + GROUP_DATA + IND
#define L_DATA_ind              LL + DATA_CO    + IND
#define L_BROADCAST_ind         LL + BROADCAST  + IND

#define L_GROUP_DATA_conf       LL + GROUP_DATA + CON
#define L_DATA_conf             LL + DATA_CO    + CON
#define L_BROADCAST_conf        LL + BROADCAST  + CON

// NL
#define N_GROUP_DATA_requ		NL + GROUP_DATA + REQ
#define N_DATA_requ			    NL + DATA_CO    + REQ
#define N_BROADCAST_requ		NL + BROADCAST  + REQ

#define N_GROUP_DATA_ind		NL + GROUP_DATA + IND
#define N_DATA_ind			    NL + DATA_CO    + IND
#define N_BROADCAST_ind			NL + BROADCAST  + IND

#define N_GROUP_DATA_conf		NL + GROUP_DATA + CON
#define N_DATA_conf			    NL + DATA_CO    + CON
#define N_BROADCAST_conf		NL + BROADCAST  + CON

// TL
#define T_GROUP_DATA_requ		TL + GROUP_DATA + REQ
#define T_DATA_requ			    TL + DATA_CO    + REQ
#define T_DEV_DATA_requ			TL + DATA_CL    + REQ
#define T_BROADCAST_requ		TL + BROADCAST  + REQ

#define T_GROUP_DATA_ind		TL + GROUP_DATA + IND
#define T_DATA_ind			    TL + DATA_CO    + IND
#define T_DEV_DATA_ind			TL + DATA_CL    + IND
#define T_BROADCAST_ind			TL + BROADCAST  + IND

#define T_GROUP_DATA_conf		TL + GROUP_DATA + CON
#define T_DATA_conf			    TL + DATA_CO    + CON
#define T_DEV_DATA_conf 		TL + DATA_CL    + CON
#define T_BROADCAST_conf		TL + BROADCAST  + CON

#define CONNECTION_TIMEOUT_ind  TL + CONNECTION_TIMEOUT  + IND
#define ACKNOWLEDGE_TIMEOUT_ind TL + ACKNOWLEDGE_TIMEOUT + IND
#define DISCONNECT_ind          TL + DISCONNECT + IND

// AL
#define A_VALUE_WRITE_requ		AL + VALUE_WRITE+ REQ
#define A_VALUE_READ_requ		AL + VALUE_READ + REQ
#define A_VALUE_RESP_requ		AL + VALUE_RESP + REQ

#define A_VALUE_WRITE_ind		AL + VALUE_WRITE+ IND
#define A_VALUE_READ_ind		AL + VALUE_READ + IND
#define A_VALUE_RESP_ind		AL + VALUE_RESP + IND

#define A_VALUE_WRITE_conf		AL + VALUE_WRITE+ CON
#define A_VALUE_READ_conf		AL + VALUE_READ + CON
#define A_VALUE_RESP_conf		AL + VALUE_RESP + CON

//-----------------------------------------------------------------------------------------------------------
// taskswitch
//-----------------------------------------------------------------------------------------------------------
#define TSIMK WTIMK
#define    EN_TASKSWITCH()   TSIMK = 0
#define   DIS_TASKSWITCH()   TSIMK = 1
#define FORCE_TASKSWITCH()   WTM0 = 0; WTM0 = 1; WTIIF = 1; EN_TASKSWITCH();        \
                             __no_operation(); __no_operation(); __no_operation();  \
                             __no_operation(); __no_operation(); __no_operation();  \
                             __no_operation(); __no_operation(); __no_operation();

//-----------------------------------------------------------------------------------------------------------
// GetEIBState
//-----------------------------------------------------------------------------------------------------------
#define EIB_LINE_DOWN       0x00
#define EIB_LINE_BOOTING    0x01
#define EIB_LINE_RUNNING    0x02

//-----------------------------------------------------------------------------------------------------------
// debug pins
//-----------------------------------------------------------------------------------------------------------
#ifdef DEBUG_PINS
#define TESTPINS_OUTPUT()   PM4 = 0x00
#define TESTPIN1_ON()       P4_bit.no0 = 1
#define TESTPIN1_OFF()      P4_bit.no0 = 0
#define TESTPIN1_TOGGLE()   P4_bit.no0 = ~P4_bit.no0
#define TESTPIN2_ON()       P4_bit.no1 = 1
#define TESTPIN2_OFF()      P4_bit.no1 = 0
#define TESTPIN2_TOGGLE()   P4_bit.no1 = ~P4_bit.no1
#define TESTPIN3_ON()       P4_bit.no2 = 1
#define TESTPIN3_OFF()      P4_bit.no2 = 0
#define TESTPIN3_TOGGLE()   P4_bit.no2 = ~P4_bit.no2
#define TESTPIN4_ON()       P4_bit.no3 = 1
#define TESTPIN4_OFF()      P4_bit.no3 = 0
#define TESTPIN4_TOGGLE()   P4_bit.no3 = ~P4_bit.no3
#else
#define TESTPINS_OUTPUT()
#define TESTPIN1_ON()
#define TESTPIN1_OFF()
#define TESTPIN1_TOGGLE()
#define TESTPIN2_ON()
#define TESTPIN2_OFF()
#define TESTPIN2_TOGGLE()
#define TESTPIN3_ON()
#define TESTPIN3_OFF()
#define TESTPIN3_TOGGLE()
#define TESTPIN4_ON()
#define TESTPIN4_OFF()
#define TESTPIN4_TOGGLE()
#endif

enum SPISpeed                                   //!< baud rates for spi
{
    SPI_BAUD_300        = 0,
    SPI_BAUD_600        = 1,
    SPI_BAUD_1200       = 2,
    SPI_BAUD_2400       = 3,
    SPI_BAUD_4800       = 4,
    SPI_BAUD_9600       = 5,
    SPI_BAUD_19200      = 6,
    SPI_BAUD_38400      = 7,
    SPI_BAUD_48000      = 8,
    SPI_BAUD_135000     = 9
};

enum PropertyInternalType                       //!< the internal memory types of a property
{
    IsNormalMem,
    IsParamVal,
    IsOTPNormalMem,
    IsOTPParamVal
};

typedef enum                                    //!< conversion speed for adc
{
    ADCHIGH,
    ADCMED,
    ADCLOW
} ADCSpeedModeType;

typedef enum                                    //!< speed for pwm
{
    PWMHIGH,
    PWMMED,
    PWMLOW,
    PWMULOW
} PWMSpeedModeType;

typedef enum                                    //!< pwm channel
{
    PWMCHAN0,
    PWMCHAN1
} PWMChannelType;

typedef enum                                    //!< pwm polarity
{
    HIGHACTIVE,
    LOWACTIVE
} PWMPolType;

typedef unsigned char       u08;
typedef unsigned short      u16;
typedef unsigned long       u32;
typedef unsigned char       BOOL;
typedef unsigned short      USHORT;
typedef unsigned char       BYTE;
typedef unsigned short      WORD;
typedef unsigned long       ULONG;

typedef struct MESSAGE                          //!< message structure
{	
    struct MESSAGE *pNext;                      //!< pointer to next message
    BYTE           Length;		                //!< length of the message
    BYTE           ServiceCode;                 //!< service code of the message
    BYTE           ASAP;
    BYTE           TSAP;
    BYTE           Data[];
} MESSAGE;

typedef MESSAGE*  MESSAGEQUEUE;                 //!< definition of a message queue
typedef MESSAGE** DYNMESSAGEQUEUE;              //!< definition of a dynamic message queue

typedef struct                                  //!< static message queues structure
{
    MESSAGEQUEUE    LinkLayer;
    MESSAGEQUEUE    NetworkLayer;
    MESSAGEQUEUE    TransportLayerGroup;
    MESSAGEQUEUE    TransportLayerConInd;
    MESSAGEQUEUE    TransportLayerConReq;
    MESSAGEQUEUE    ApplicationLayer;
    MESSAGEQUEUE    ManagementLayer;
} STATICMESSAGEQUEUES;

typedef struct                                  //!< dynamic message queues structure
{
    DYNMESSAGEQUEUE    LinkLayer;
    DYNMESSAGEQUEUE    NetworkLayer;
    DYNMESSAGEQUEUE    TransportLayerGroup;
    DYNMESSAGEQUEUE    TransportLayerConInd;
    DYNMESSAGEQUEUE    TransportLayerConReq;
    DYNMESSAGEQUEUE    ApplicationLayer;
    DYNMESSAGEQUEUE    ManagementLayer;
} DYNMESSAGEQUEUES;

typedef void(*pIntFunc)();                      //!< function pointer for COM0 send and receive interrupts

typedef struct PROPERTY                         //! structure for a property
{
    BYTE PropertyID;                            //!< possible: 1...255
    BYTE PropertyType;                          //!< defines type and writeability
    USHORT MaxLength;                           //!< number of max. elements
    USHORT *pActualLength;                      //!< pointer to actual numbers of elem or NULL pointer if fix length
    const void *pData;                          //!< void pointer to data(s) or function
    enum PropertyInternalType InternalType;     //!< indicates if property is normal memory, eeval or otp
} PROPERTY;

typedef struct INTERFACE_OBJECT                 //! structure for a interface object
{
    USHORT ObjType;                             //!< object type (2 byte: higbyte,lowbyte)
    USHORT PropCount;                           //!< max 256 properties possible
    PROPERTY *PropTable;                        //!< pointer to propertytable (first property in table!)
} INTERFACE_OBJECT;

typedef struct INTERFACE_ROOT                   //! root of the interface objects
{
    USHORT ObjCount;                            //!< max 65536 objects possible
    const INTERFACE_OBJECT *ObjTable;           //!< pointer to objectable
} INTERFACE_ROOT;

typedef struct                                  //!< timer structure
{
    USHORT Ctrl;                                //!< flags: TMR_USED,TMR_EXPIRED,TMR_AUTOFREE
    ULONG Limit;                                //!< time when timer will be expired
} TIMER;

typedef struct                                  //!< timer table structure
{
    BYTE      NumOfTimers;
    TIMER     Timers[NUM_OF_TIMERS];
} TIMER_TAB;

typedef struct DEBOUNCEKIT                      //!< debounce kit structure
{
    USHORT      LasValue;                       // last sample
    USHORT      DebValue;                       // last debounced value
    USHORT      DebTime;
} DEBOUNCEKIT;

typedef struct                                  //!< parameter value structure
{
    BYTE    ValID;
    BYTE    Length;
    BYTE    ValLength;
    BYTE    NotUsed;
} ParamVal;

typedef struct                                  //!< parameter sector structure
{
    BYTE    Data[SECTOR_SIZE - (2 * sizeof(u32))];
    u32     SectorActivNum2;
    u32     SectorActivNum1;
} ParamSector;

typedef struct                                  //!< parameter management structure
{
    const ParamSector*  ParamSectors;
    BYTE*               pNextFreePos;
    BYTE                ActiveSector;
    BYTE                NumOfEntries;
    BOOL                ReorgIsActive;
    BYTE                Bank;
} ParamMgmt;

typedef struct                                  //!< property parameter value structure
{
    ParamMgmt* pPSM;
    BYTE ValID;
    BYTE Offset;
} PROPERTYPARMVAL;

typedef BYTE(*pFunc)(MESSAGE* pMsg, void* pParam);  //!< function pointer for property functioncall

typedef struct                                  //!< property functioncall structure
{
    pFunc   Callback;                           // Callback function
    void*   pParam;                             // pointer to parameter
} FUNCTIONCALL;

typedef void* CObjPtr;

typedef struct                                  //!< structure of the application info block
{
    USHORT                  AIBVersion;
    BYTE ApplFirmwareVersion;
    BYTE ApplFirmwareSubVersion;
    void (*AppMain)         (void);
    void (*AppSave)         (void);
    void (*AppUnload)       (void);
    const CObjPtr*          pCObjects;
    BYTE*                   pRAMFlags;
    TIMER_TAB*              pUserTimerTab;
    const INTERFACE_ROOT*   pUsrIntObjRoot;
    ParamMgmt*              pUsrParamMgmt;
    USHORT                  WatchDogTime;
} AppInfoBlock;

typedef struct JUMPTABLE                        //!< structure of the jump table
{
    // debounce api
    void        (*_DebounceInit)        (DEBOUNCEKIT* kit, USHORT initvalue);
    void        (*_Debounce)            (USHORT sample, USHORT mask, DEBOUNCEKIT* kit, USHORT debouncetime);

    // al api
    BOOL        (*_TestAndCopyObject)   (USHORT objectNr, void* dst, BYTE len);
    BOOL        (*_SetAndTransmitObject)(USHORT objectNr, void* src, BYTE len);
    BOOL        (*_TestObject)          (USHORT objectNr);
    BOOL        (*_TransmitObject)      (USHORT objectNr);
    BOOL        (*_ReadObject)          (USHORT objectNr);
    BYTE        (*_SetRAMFlags)         (BYTE objectNr, BYTE flags);

    // timer api
    ULONG       (*_GetSystemTime)       (void);
    void        (*_TmInit)              (BYTE NumOfTimers);
    void        (*_TmStart)             (TIMER* pTimer, ULONG ticks);
    void        (*_TmAddStart)          (TIMER* pTimer, ULONG ticks);
    BOOL        (*_TmIsExpired)         (TIMER* pTimer);
    BOOL        (*_TmIsRunning)         (TIMER* pTimer);
    void        (*_TmStop)              (TIMER* pTimer);

    // adc api
    void        (*_ADCInit)             (ADCSpeedModeType speed);
    void        (*_ADCShutdown)         (void);
    USHORT      (*_ADCRead)             (BYTE port, BYTE ADCcount);
    void        (*_ADCStop)             (void);
    BOOL        (*_ADCIsInterrupted)    (void);
    void        (*_ADCResetInterrupted) (void);
    BYTE        (*_CalcPEIType)         (BYTE adval);

    // pwm api
    void        (*_PWMInit)             (PWMChannelType channel, PWMPolType mode, PWMSpeedModeType speed);
    void        (*_PWMStop)             (PWMChannelType channel);
    void        (*_PWMSetValue)         (PWMChannelType channel, BYTE value);

    // ft12 api
    void        (*_FT12Init)            (USHORT TimeoutTime, BYTE len, BYTE* rcvBuffer, BYTE* trmBuffer, BYTE baud, BYTE config);
    BOOL        (*_FT12Send)            (BYTE* src, BYTE len, BYTE* result);
    BOOL        (*_FT12Get)             (BYTE* dst, BYTE* len);

    // handshake api
    void        (*_HSInit)              (USHORT TimeoutTime, BYTE len, BYTE* rcvBuffer, BYTE* trmBuffer, BYTE baud, BYTE config);
    BOOL        (*_HSSetFrame)          (BYTE* src, BYTE len, BYTE* pResult);
    BOOL        (*_HSGetFrame)          (BYTE* dst, BYTE* len);

    // spi
    void        (*_SPIInit)             (enum SPISpeed speed, BYTE CKPDAP, BOOL MSBFirst);
    BOOL        (*_SPISend)             (BYTE* pData, BYTE length);

    // flash api
    BOOL        (*_FlashSegErase)       (u32 dst);
    BOOL        (*_FlashWrite)          (u32 dst, void* src, BYTE count);
    BOOL        (*_FlashIsBlockBlank)   (BYTE* pVal, u16 count);

    // parameter api
    BOOL        (*_ParamInitVal)        (BYTE* src, BYTE ValID, BYTE ValLength);
    BOOL        (*_ParamReadVal)        (BYTE ValID, void* dst, BYTE len);

    // msg functions
    MESSAGE*    (*_MsgCreate)           (void);
    void        (*_MsgDiscard)          (MESSAGE *pMsg);
    MESSAGE*    (__monitor *_MsgGet)    (MESSAGEQUEUE* pQueue);
    void        (__monitor *_MsgPost)   (MESSAGE *pMsg, MESSAGEQUEUE* pQueue);
    void        (__monitor *_MsgUndoGet)(MESSAGE *pMsg, MESSAGEQUEUE* pQueue);
    void        (*_MsgSwitchQueue)      (DYNMESSAGEQUEUE* dynq, MESSAGEQUEUE* staticq);
    void        (*_MsgResetDynQueues)   (void);

    // utility
    BOOL        (*_IsApplicationLoaded) (void);
    BYTE        (*_Dummy01)             (void);
    void        (*_GetPhysAddr)         (BYTE* dst);
    void        (*_GetSerialNumber)     (BYTE* dst);
    BOOL        (*_ReadBCU2Adr100)      (BYTE offset, BYTE count, void* dst);
    void        (*_TriggerWatchDog)     (void);

    // interrupts
    void        (*_IntRegister)         (pIntFunc func, BYTE IntAddr);
    void        (*_IntUnregister)       (BYTE IntAddr);
    void        (*_IntResetAll)         (void);

    // memory
    DYNMESSAGEQUEUES*                   _pDynQueues;
    STATICMESSAGEQUEUES*                _pStaticQueues;
} JUMPTABLE;

#define PID_OBJECT_TYPE                             1



extern const JUMPTABLE U;
void __program_start(void);

#endif
