/******************************************************************************
*
* File: main.c
*
* Description:
* This template shows how to initialize the FT12 driver to have the possibility
* of sending and receiving frames via FT12.
* The operating system of the BIM M 13x devices does not support the
* pei switch mechanism of the BIM M 113. In fact a user application
* program has to be written that activates the FT12 driver, posts and
* gets messages to and from linklayer.
* This template shows how this could be done.
* The template is completely free in use and can be adapted to your own requirements.
* There is no guaranty that it will work with your hardware and software.
*
* Additional commentaries:
* ========================
*
* Hardware information:
* ---------------------
* fPRS = 9,830,400 Hz
*
* Relevant pins on BIM M13x EVB:
* ------------------------------
*                                        +---+
*                                        |   |
* +--------------------------------------+   +-------------------------------------+
* |                                                                                |
* |  25     23    21    19    17    15    13     11     9     7     5     3     1  |
* |                                                                                |
* |  26  24(GND)  22    20    18    16  14(TxD)  12  10(RxD)  8     6     4     2  |
* |      =======                        =======      =======                       |
* +---------|------------------------------|------------^--------------------------+
*           |                              |            |
*           -                              V            |
*
*
* Additional info:
* ----------------
* After executing FT12Init(...) the BIM M 13x is able to receive and transmit
* frames via FT12. That means if a valid FT12 frame was received the bim
* automatically sends an ACK to the transmitter.
* The received frame is stored in the "rcvBuffer" that was specified at FT12Init.
* If there is a FT12 frame in the "rcvBuffer" the FT12Get(...) function copies
* the payload to the specified "dst" (destination) and writes the number of copied
* bytes to "len".
* The FT12Send function first checks if there is still data in the "trmBuffer"
* and returns "FALSE" if this is the case.
* Otherwise it copies "len" bytes from "src" (source) to the "trmBuffer".
* The "src" data means the payload of the FT12 frame.
* The len, checksum and FT12 VarFrame informations are added automatically.
* FT12Get and FT12Send are only for FT12 frames with variable length.
* FT12 frames with fix length are handled automatically by the bim.
*
* If an ACK does not arrive, the BIM automatically repeats the frame.
*
* A second call of "FT12Send" returns FALSE if the previous send request
* is not finished yet. The buffer is cleared after receiving the ACK or after
* repeating the frame 3 times without ack.
*
* Example:
* --------
*
* group telegram with 14 bytes on eib:
* bc 11 ff 56 78 ef 00 80 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e
*
* add information that this telegramm is an indication (and not a confirm):
* 29 bc 11 ff 56 78 ef 00 80 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e
*
* send data via FT12 (len = 23 (1 + 8 + 14))
* after executing FT12Send the "trmBuffer" contains the following data (30 bytes):
* 68 18 18 68 d3 29 bc 11 ff 56 78 ef 00 80 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e 6e 16
*
*
* frame via FT12:
* 68 18 18 68 73 11 BC 00 00 12 34 EF 00 80 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 5E 16
*
* after receiving the FT12 frame the following data is in the "rcvBuffer" (24 bytes):
* 17 11 bc 00 00 12 34 ef 00 80 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e
* the first byte is the number of bytes
*
* after executing FT12Get:
* len = 0x17
* dst = 11 bc 00 00 12 34 ef 00 80 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e
*
* ****************************************************************************
*
* Date: 07/17/2007
*
* Revision: 2
*
* (C) 2007 Copyright Siemens AG
*
* Author: Schiekofer
*
* Modified: 08/01/2008
*           
* Changes:
*           - FT12Patch is used
******************************************************************************/
#include "BIM_M13x.h"
#include "BIM_M13x_FT12Patch.h"

#define FT12SENDCOUNT               5       // max try count for FT12 send

// service codes as they are in BCU 2
#define SC_L_DATA_REQ               0x11
#define SC_L_DATA_CON               0x2E
#define SC_L_DATA_IND               0x29

BYTE RAMFlags[(NUM_OF_COM_OBJ / 2) + 1];

// buffers that are needed by the FT12 driver
BYTE InBuffer[30];
BYTE OutBuffer[30];

// a message queue where messages from linklayer should be posted
MESSAGEQUEUE UserQueue1 = NULL;

// the result for the FT12 send opeation
BYTE FT12res;

// this function copies the bytes of a message one position to left
void ModifyMsgL(MESSAGE* pMsg)
{
    BYTE i;

    pMsg->Length--;
    pMsg->ServiceCode = pMsg->Data[0];
    for (i = 0; i < pMsg->Length; i++)
    {
        pMsg->Data[i] = pMsg->Data[i + 1];
    }
}

// this function copies the bytes of a message one position to right
void ModifyMsgR(MESSAGE* pMsg)
{
    BYTE i;

    for (i = pMsg->Length; i > 0; i--)
    {
        pMsg->Data[i] = pMsg->Data[i - 1];
    }
    pMsg->Data[0] = pMsg->ServiceCode;
    pMsg->Length++;
}

// this function checks if the message is a linklayer request and posts
// the message to the linklayer
void HandleLinkLayerRequest(MESSAGE* pMsg)
{
    if (pMsg->Data[0] != SC_L_DATA_REQ)
    {
        U._MsgDiscard(pMsg);
        return;
    }

    ModifyMsgL(pMsg);
    pMsg->ServiceCode = (LL + REQ);
    U._MsgPost(pMsg, &U._pStaticQueues->LinkLayer);
}

// this function sends a message over FT12
void SendMessage(MESSAGE* pMsg)
{
    ULONG time;
    USHORT count = FT12SENDCOUNT;

    ModifyMsgR(pMsg);
    while ( (U._FT12Send(pMsg->Data, pMsg->Length, NULL) == FALSE) && (count-- != 0x00) )
    {
        time = U._GetSystemTime();
        // wait a few msecs before try again
        while ((U._GetSystemTime() - time) < MSEC3)
        {
            FORCE_TASKSWITCH();
        }
    }
}

// this function clears the specified message queue
void ClearMsgQueue(MESSAGEQUEUE mq)
{
    MESSAGE* pMsg;
    while ((pMsg = U._MsgGet(&mq)) != NULL)
    {
        U._MsgDiscard(pMsg);
    }
}

void AppInit(void)
{
    // init the FT12 driver  (19200 bits/s, even parity)
    // baud = (register) BRGC0    (19,200 bits/s -> baud = 0xC8)
    // config = (register) ASIM0  (even parity and one stop bit -> config = 0xFD)
    FT12InitPatch(MSEC26, 30, InBuffer, OutBuffer, 0xC8, 0xFD);

    // get all messages from linklayer in 'UserQueue1'
    U._MsgSwitchQueue(&U._pDynQueues->NetworkLayer, &UserQueue1);

    // after initialization a taskswitch must be done because
    // the task switch is disabled at cstartup
    FORCE_TASKSWITCH();
}

//------------------------------------------------------------------------
//  This is the main function of the application program
//  It is called at the end of cstartup
//------------------------------------------------------------------------
void main (void)
{
    MESSAGE* pSMsg = NULL;          // pointer to message
    MESSAGE* pRMsg = NULL;          // pointer to message
    BYTE Len;

    // Do some initializations
    AppInit();

    // Start the main loop of the application program
    // it will be interrupted every 3.3ms and
    // continued after a complete system cycle
    for(;;)
    {
        // it is possible to check the pei type resistor with the adc api
        /*****************************************************************
        U._ADCInit(ADCLOW);
        for(;;)
        {
            if (U._CalcPEIType(U._ADCRead(4, 16) >> 4) == 10)
            {
                DIS_TASKSWITCH();

                // get all messages from linklayer in 'UserQueue1'
                U._MsgSwitchQueue(&U._pDynQueues->NetworkLayer, &UserQueue1);

                EN_TASKSWITCH();
                break;
            }
            else
            {
                DIS_TASKSWITCH();

                // reset the message system of the operating system to the
                // original state
                U._MsgResetDynQueues();
                // clear the 'UserQueue1'
                ClearMsgQueue(UserQueue1);

                EN_TASKSWITCH();
            }
        }
        U._ADCShutdown();
        ******************************************************************/

        // first get a free message buffer
        if (pSMsg == NULL)
        {
            pSMsg = U._MsgCreate();
            while (pSMsg == NULL)
            {
                FORCE_TASKSWITCH();
                pSMsg = U._MsgCreate();
            }
        }

        // is there a message from FT12 to send to eib
        if (U._FT12Get(pSMsg->Data, &Len) == TRUE)
        {
            pSMsg->Length = Len;
            HandleLinkLayerRequest(pSMsg);
            pSMsg = NULL;
        }

        // is there a message from eib to send over FT12
        pRMsg = U._MsgGet(&UserQueue1);
        if (pRMsg != NULL)
        {
            if ((pRMsg->ServiceCode & (LAYER_MASK + TYPE_MASK)) == (LL + IND))
            {
                pRMsg->ServiceCode = SC_L_DATA_IND;
                SendMessage(pRMsg);
            }
            else if ((pRMsg->ServiceCode & (LAYER_MASK + TYPE_MASK)) == (LL + CON))
            {
                pRMsg->ServiceCode = SC_L_DATA_CON;
                SendMessage(pRMsg);
            }
            U._MsgDiscard(pRMsg);
        }
    }
}

//------------------------------------------------------------------------
//  This function is called on loss of power
//------------------------------------------------------------------------
void save()
{
}

//------------------------------------------------------------------------
//  This function is called if there was an
//  unload of the application program
//------------------------------------------------------------------------
void unload()
{
}

const BYTE NbComsOb = NUM_OF_COM_OBJ;
const CObjPtr CObjects[] =      // set table for com objects
{
    (void*)&NbComsOb,           // ptr to number of cobjects
//------------------------------------------------------------------------
//  Here the addresses of the communication objects are specified
//------------------------------------------------------------------------
};

//------------------------------------------------------------------------
//  In the application info block the application program gives some
//  necessary information for the operation system
//------------------------------------------------------------------------
#pragma constseg = APPINFOBLOCK
const AppInfoBlock AIB =
{
    Swap(0x0001),                   // AIBVersion
    0x01, 0x01,                     // ApplFirmwareVersion, ApplFirmwareSubVersion
    __program_start,                // AppMain
    save,                           // AppSave
    unload,                         // AppUnload
    CObjects,                       // pCObjects
    RAMFlags,                       // pRAMFlags
    NULL,                           // pUserTimerTab
    NULL,                           // pUsrIntObjRoot
    NULL,                           // pUsrParamMgmt
    0x0000                          // WatchDogTime
};
#pragma constseg = default
